// Program:  glFont.c - OpenGL FreeType Wrapper
// Language: C++
// Author:   Daniel Kennedy (Syn9)
//
// Copyright (c) 2009-2012
// ____________________________________________________________________________
#include "glFont.h"

namespace basecode
{

	glFont::~glFont() { clear(); }

	void glFont::clear ()
	{
		glDeleteLists(baseList, numChars);
		glDeleteTextures(numChars, texture);
		delete [] fchars;
		delete [] texture;

		baseList = 0;
		numChars = 0;
		topRow = 0;
	}
	

	void glFont::createDisplayList (unsigned int ch)
	{
		GLfloat w0, h0, l0, t0, xw, yw, ww, hw;
		
		glNewList(baseList + ch, GL_COMPILE);

			if (format == glFont::TTF)
			{
				glBindTexture(GL_TEXTURE_2D, texture[ch]);
				w0 = (GLfloat) fchars[ch].width;
				h0 = (GLfloat) fchars[ch].height;
				l0 = (GLfloat) fchars[ch].left;
				t0 = (GLfloat) fchars[ch].top;

				glBegin(GL_QUADS);
				
					glTexCoord2f(0, 1);
					glVertex2f(l0, -t0);

					glTexCoord2f(0, 0);
					glVertex2f(l0, -t0 + h0);

					glTexCoord2f(1, 0);
					glVertex2f(l0 + w0, -t0 + h0);

					glTexCoord2f(1, 1);
					glVertex2f(l0 + w0, -t0);

				glEnd();

				glTranslatef((float)fchars[ch].advance / 64, 0, 0);

			} else {

				if (fchars[ch].id == 32)
				{
					glTranslatef((float)fchars[ch].advance, 0, 0);
					glEndList();
					return;
				}

				glBindTexture(GL_TEXTURE_2D, texture[fchars[ch].page]);
				w0 = (float)fchars[ch].width;
				h0 = (float)fchars[ch].height + .5f;
				l0 = (float)fchars[ch].left;
				t0 = (float)fchars[ch].top;
				xw = fchars[ch].xw;
				yw = fchars[ch].yw;
				ww = w0 / (float)txWidth;
				hw = h0 / (float)txHeight;
				
				glBegin(GL_QUADS);
				
					glTexCoord2f(xw, yw);
					glVertex2f(l0, -topRow + t0 + 1);

					glTexCoord2f(xw, yw - hw);
					glVertex2f(l0, -topRow + t0 + h0);

					glTexCoord2f(xw + ww, yw - hw);
					glVertex2f(l0 + w0, -topRow + t0 + h0);

					glTexCoord2f(xw + ww, yw);
					glVertex2f(l0 + w0, -topRow + t0 + 1);

				glEnd();
				
				glTranslatef((float)fchars[ch].advance, 0, 0);
			}
			

		glEndList();
	}


	void glFont::createTexture (FT_Face face, unsigned int ch)
	{
		int w0, h0, pitch;
		w0 = face->glyph->bitmap.width;
		h0 = face->glyph->bitmap.rows;
		pitch = face->glyph->bitmap.pitch;

		fchars[ch].left = face->glyph->bitmap_left;
		fchars[ch].top = face->glyph->bitmap_top;
		fchars[ch].advance = face->glyph->advance.x;

		if (fchars[ch].top > topRow) topRow = fchars[ch].top;

		int useBlank=0;
		if (w0 == 0 || h0 == 0 || pitch == 0) useBlank = 1;

		int w1=1, h1=1;
		while (w1 < w0) w1 *= 2;
		while (h1 < h0) h1 *= 2;

		fchars[ch].width = w1;
		fchars[ch].height = h1;

		int glBufferSize = w1 * h1 * 4;
		int glPitch = w1 * 4;
		GLubyte *glBuffer;
		glBuffer = (GLubyte*) calloc(glBufferSize, sizeof(GLubyte));

		glBindTexture(GL_TEXTURE_2D, texture[ch]);

		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		//glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		//glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

		int row, col, glRow, glCol;
		GLubyte bt = 0;
		
		for (row = 0; row < h0; row++)
		{
			for (col = 0; col < w0; col++)
			{
				if (useBlank == 0)
					bt = face->glyph->bitmap.buffer[row * pitch + col];

				glRow = (h1 - row - 1) * glPitch;
				glCol = col * 4;
				
				glBuffer[glRow + glCol + 0] = 255;
				glBuffer[glRow + glCol + 1] = 255;
				glBuffer[glRow + glCol + 2] = 255;
				glBuffer[glRow + glCol + 3] = bt;
			}
		}

		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, w1, h1, 0, GL_RGBA, GL_UNSIGNED_BYTE, glBuffer);

		free(glBuffer);
	}

	void glFont::draw (string text)
	{
		glEnable(GL_TEXTURE_2D);
		glPushMatrix();

			glTranslatef(0, (float)topRow, 0);

			glListBase(baseList);
			glCallLists(text.size(), GL_UNSIGNED_BYTE, text.c_str());

		glPopMatrix();
		glDisable(GL_TEXTURE_2D);
	}

	int glFont::getHeight (std::string text)
	{
		int txtlen = text.length();
		int i, ch, h0, maxH;

		maxH = topRow;

		char * t = (char*)text.c_str();

		if (format == glFont::BMFNT)
		{
			for (i = 0; i < txtlen; i++)
			{
				ch = t[i];
				h0 = topRow - fchars[ch].top + fchars[ch].height;
				if (h0 > maxH) maxH = h0;
			}
			return maxH;
		}
		for (i = 0; i < txtlen; i++)
		{
			ch = t[i];
			h0 = topRow - fchars[ch].top + fchars[ch].height;
			if (h0 > maxH) maxH = h0;
		}

		return maxH;
	}


	int glFont::getHeight (glfont_str_t text)
	{
		int txtlen = text.length();
		int i, ch, h0, maxH;

		maxH = topRow;

		glfont_char_t * t = (glfont_char_t*)text.c_str();

		if (format == glFont::BMFNT)
		{
			for (i = 0; i < txtlen; i++)
			{
				ch = t[i];
				h0 = topRow - fchars[ch].top + fchars[ch].height;
				if (h0 > maxH) maxH = h0;
			}
			return maxH;
		}
		for (i = 0; i < txtlen; i++)
		{
			ch = t[i];
			h0 = topRow - fchars[ch].top + fchars[ch].height;
			if (h0 > maxH) maxH = h0;
		}

		return maxH;
	}


	int glFont::getWidth (std::string text)
	{
		int txtlen = text.length();
		int i, ch, w0 = 0;
		
		char * t = (char*)text.c_str();

		if (format == glFont::BMFNT)
		{
			for (i = 0; i < txtlen; i++)
			{
				ch = t[i];
				w0 += fchars[ch].advance;
			}
			return w0;
		}
		
		for (i = 0; i < txtlen; i++)
		{
			ch = t[i];
			w0 += fchars[ch].advance / 64;
		}

		return w0;
	}

	
	int glFont::getWidth (glfont_str_t text)
	{
		int txtlen = text.length();
		int i, ch, w0 = 0;
		
		glfont_char_t * t = (glfont_char_t*)text.c_str();

		if (format == glFont::BMFNT)
		{
			for (i = 0; i < txtlen; i++)
			{
				ch = t[i];
				w0 += fchars[ch].advance;
			}
			return w0;
		}
		
		for (i = 0; i < txtlen; i++)
		{
			ch = t[i];
			w0 += fchars[ch].advance / 64;
		}

		return w0;
	}

	
	bool glFont::load (string filename)
	{
		return load(filename, 16);
	}

	bool glFont::load (string filename, int fsize)
	{
		string ext = filename.substr(filename.size() - 3, 3);

		if (ext.compare("ttf") == 0) return load_ttf(filename, fsize);
		if (ext.compare("fnt") == 0) return load_bmfnt(filename);

		return false;
	}

	bool glFont::load_bmfnt (string filename)
	{
		printf("Loading Font: %s\n", filename.c_str());
		
		format = glFont::BMFNT;
		if (numChars > 0) clear();

		std::string directory = filename.substr(0, filename.find_last_of("/") + 1);
		
		fstream fs;

		char str[1024];
		vector<string> filelines;
		string temp, spc = " ";
		size_t si;

		fs.open(filename.c_str(), std::fstream::in);
		if (!fs.is_open()) return false;
		
		fs.getline(str, 1024, '\n');
		while (fs.gcount() > 0)
		{
			temp = string(str);
			do
			{
				si = temp.find("  ");
				if (si == string::npos) break;
				temp.replace(si, 2, spc);
			} while (true);
			
			filelines.push_back(temp);
			fs.getline(str, 1024, '\n');
		}

		fs.close();

		unsigned int totallines = filelines.size();

		std::vector<std::string> expArray;
		std::vector<std::string> expArray2;

		// common lineHeight=20 base=17 scaleW=64 scaleH=64 pages=6 packed=0 alphaChnl=4 redChnl=0 greenChnl=0 blueChnl=0
		expArray.clear();
		str_explode(filelines[1], " ", &expArray);

		topRow = atoi(expArray[2].substr(expArray[2].find("=") + 1, 8).c_str());
		txWidth = atoi(expArray[3].substr(expArray[3].find("=") + 1, expArray[3].size()).c_str());
		txHeight = atoi(expArray[4].substr(expArray[4].find("=") + 1, expArray[4].size()).c_str());

		numPages = atoi(expArray[5].substr(expArray[5].find("=") + 1, 8).c_str());
		unsigned int p;
		string s;

		vector<string> texList(numPages);

		texture = new GLuint[numPages];
		glGenTextures(numPages, texture);		

		for (unsigned int i = 2; i < 2 + numPages; i++)
		{
			// page id=0 file="pf_ronda_seven_bold_0.tga"
			expArray.clear();
			str_explode(filelines[i], " ", &expArray);

			p = atoi(expArray[1].substr(expArray[1].find("=") + 1, expArray[1].size()).c_str());
			s = expArray[2].substr(expArray[2].find("=") + 2, expArray[2].size());
#ifdef _WINDOWS
			s = directory + s.substr(0, s.size() - 1);
#else
			s = directory + s.substr(0, s.size() - 2);
#endif
			texList[p] = s;
			printf("Loading Texture: %s\n", s.c_str());

			glBindTexture(GL_TEXTURE_2D, texture[p]);
			glfwLoadTexture2D(s.c_str(), 0);
			//glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
			//glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
			//glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
			//glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
			glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

		}

		//chars count=189
		expArray.clear(); p = 2 + numPages;
		str_explode(filelines[p], " ", &expArray);

		numChars = atoi(expArray[1].substr(expArray[1].find("=") + 1, expArray[1].size()).c_str()) + 1; // add 1 because first character cant be index 0
		fchars = new glFontCharacter[numChars];
		baseList = glGenLists(numChars);
				
		int c = 1;
		
		fchars[0].id = 0;

		for (unsigned int i = ++p; i < totallines; i++)
		{
			//char id=32   x=62    y=47    width=1     height=1     xoffset=0     yoffset=17    xadvance=6     page=0  chnl=15
			expArray.clear();
			str_explode(filelines[i], " ", &expArray);
			if (expArray[0].compare("char") != 0) continue;

			fchars[c].id = atoi(expArray[1].substr(expArray[1].find("=") + 1, expArray[1].size()).c_str());
			fchars[c].x = (float)atoi(expArray[2].substr(expArray[2].find("=") + 1, expArray[2].size()).c_str());
			fchars[c].y = (float)atoi(expArray[3].substr(expArray[3].find("=") + 1, expArray[3].size()).c_str());
			fchars[c].width = atoi(expArray[4].substr(expArray[4].find("=") + 1, expArray[4].size()).c_str());
			fchars[c].height = atoi(expArray[5].substr(expArray[5].find("=") + 1, expArray[5].size()).c_str());
			fchars[c].left = atoi(expArray[6].substr(expArray[6].find("=") + 1, expArray[6].size()).c_str());
			fchars[c].top = atoi(expArray[7].substr(expArray[7].find("=") + 1, expArray[7].size()).c_str());
			fchars[c].advance = atoi(expArray[8].substr(expArray[8].find("=") + 1, expArray[8].size()).c_str());

			int f = fchars[c].width + fchars[c].left + 1;
			if (fchars[c].advance > f && fchars[c].id > 255) fchars[c].advance = f;

			fchars[c].page = atoi(expArray[9].substr(expArray[9].find("=") + 1, expArray[9].size()).c_str());

			fchars[c].xw = fchars[c].x / (float)txWidth;
			fchars[c].yw = 1 - fchars[c].y / (float)txHeight;

			createDisplayList(c);
			c++;
		}

		
		return true;
	}

	bool glFont::load_ttf (string filename, int fsize)
	{
		format = glFont::TTF;
		if (numChars > 0) clear();

		FT_Library library;
		if (FT_Init_FreeType(&library)) return false;

		FT_Face face;

		numChars = 256;//face->num_glyphs;
			
		fchars = new glFontCharacter[numChars];
		texture = new GLuint[numChars];
		baseList = glGenLists(numChars);
		glGenTextures(numChars, texture);
		glEnable(GL_TEXTURE_2D);

		if (FT_New_Face(library, filename.c_str(), 0, &face)) return false;

		FT_Set_Char_Size(face, 0, fsize << 6, 96, 96);

		topRow = 0;

		for (int ch=0; ch < numChars; ch++)
		{
			if (FT_Load_Char(face, ch, FT_LOAD_RENDER)) return false;

			createTexture(face, ch);
			createDisplayList(ch);
		}

		FT_Done_Face(face);
		FT_Done_FreeType(library);

		glDisable(GL_TEXTURE_2D);
		
		return true;
	}


	std::string glFont::prep (std::string text)
	{
		std::string newstr;
		int i, a;
		int txtlen = text.length();
		
		if (format == glFont::TTF) return text;
		
		char * t = (char*)text.c_str();

		for (i = 0; i < txtlen; i++)
		{
			for (a = 0; a < numChars; a++)
			{
				if (fchars[a].id == t[i])
				{
					newstr.push_back((char)a);
					break;
				}
			}
		}
		
		return newstr;
	}

	void glFont::print (float x, float y, float xscale, float yscale, std::string text, Color c0)
	{
		glEnable(GL_TEXTURE_2D);
		glPushMatrix();

			glTranslatef(x,y,0);
			glTranslatef(0, topRow * yscale, 0);

			if (xscale != 1 && yscale == 1) glScalef(xscale, 1, 1);
			if (yscale != 1 && xscale == 1) glScalef(1, yscale, 1);
			if (xscale != 1 && yscale != 1) glScalef(xscale, yscale, 1);

			glColor4fv((GLfloat*) &c0);
			
			glListBase(baseList);
			glCallLists(text.size(), GL_UNSIGNED_BYTE, text.c_str());

		glPopMatrix();
		glDisable(GL_TEXTURE_2D);
	}


	void glFont::print (float x, float y, float scale, std::string text, Color c0)
	{
		print(x, y, scale, scale, text, c0);
	}


	void glFont::print (float x, float y, std::string text, Color c0)
	{
		print(x, y, 1, 1, text, c0);
	}


	void glFont::print (float scale, std::string text, Color c0)
	{
		print(0, 0, scale, scale, text, c0);
	}


	void glFont::print (std::string text, Color c0)
	{
		print(0, 0, 1, 1, text, c0);
	}


	glfont_str_t glFont::prep (glfont_str_t text)
	{
		glfont_str_t newstr;
		int i, a;
		int txtlen = text.length();
		
		if (format == glFont::TTF) return text;
		
		glfont_char_t * t = (glfont_char_t*)text.c_str();

		for (i = 0; i < txtlen; i++)
		{
			for (a = 0; a < numChars; a++)
			{
				if (fchars[a].id == t[i])
				{
					newstr.push_back((glfont_char_t)a);
					break;
				}
			}
		}
		
		return newstr;
	}

	void glFont::print (float x, float y, float xscale, float yscale, glfont_str_t text, Color c0)
	{
		glEnable(GL_TEXTURE_2D);
		glPushMatrix();

			glTranslatef(x,y,0);
			glTranslatef(0, topRow * yscale, 0);

			if (xscale != 1 && yscale == 1) glScalef(xscale, 1, 1);
			if (yscale != 1 && xscale == 1) glScalef(1, yscale, 1);
			if (xscale != 1 && yscale != 1) glScalef(xscale, yscale, 1);

			glColor4fv((GLfloat*) &c0);
			
			glListBase(baseList);
			glCallLists(text.size(), glfont_gl_t, text.c_str());

		glPopMatrix();
		glDisable(GL_TEXTURE_2D);
	}


	void glFont::print (float x, float y, float scale, glfont_str_t text, Color c0)
	{
		print(x, y, scale, scale, text, c0);
	}


	void glFont::print (float x, float y, glfont_str_t text, Color c0)
	{
		print(x, y, 1, 1, text, c0);
	}


	void glFont::print (float scale, glfont_str_t text, Color c0)
	{
		print(0, 0, scale, scale, text, c0);
	}


	void glFont::print (glfont_str_t text, Color c0)
	{
		print(0, 0, 1, 1, text, c0);
	}

	void str_explode(string haystack, string needle, vector<string> *result)
	{
		std::string::size_type start, end;

		result->clear();

		start = 0;
		end = haystack.find_first_of(needle, start);
		while (end != std::string::npos)
		{
			if (end - start > 0)
			{
				result->push_back(haystack.substr(start, end - start));
			} else {
				result->push_back("");
			}
			start = end + 1;
			end = haystack.find_first_of(needle, start);
		}
		if (end == std::string::npos && haystack.length() - start > 0) result->push_back(haystack.substr(start, haystack.length() - start));
		

	}
}
