// Program:  Model.cpp - 3D Model Class
// Language: C++
// Author:   Daniel Kennedy (Syn9)
//
// Copyright (c) 2009-2010
// ____________________________________________________________________________
#include "Model.h"

namespace basecode
{
	using namespace Math;

	float Model::area2(Face t)
	{
		vec3f n;

		n = (vertices[t.point_id[1]] - vertices[t.point_id[0]]).cross(vertices[t.point_id[2]] - vertices[t.point_id[0]]);

		return n.dot(n);
	}

	Model::Model()
	{
		vertexSnapDistance = .001f;

		genBuffers();
	}


	Model::~Model()
	{
	}

	void Model::calc_normal(unsigned int face)
	{
		faces[face].normal = (vertices[faces[face].point_id[1]] - vertices[faces[face].point_id[0]]).cross(vertices[faces[face].point_id[2]] - vertices[faces[face].point_id[0]]);
		faces[face].normal.normalize();
		faces[face].d = faces[face].normal.dot(vertices[faces[face].point_id[0]]);
	}

	void Model::calc_normals()
	{
		for (unsigned int f = 0; f < faces.size(); f++)
			calc_normal(f);
	}

	void Model::center()
	{
		vec3f c = centroid();

		for (unsigned int i = 0; i < vertices.size(); i++)
		{
			vertices[i] -= c;
		}
	}

	vec3f Model::centroid()
	{
		vec3f c;

		if (vertices.size() == 0) return c;

		for (unsigned int i = 0; i < vertices.size(); i++)
		{
			c += vertices[i];
		}
		
		c /= (float)vertices.size();

		return c;
	}

	vec3f Model::fcentroid()
	{
		vec3f c;
		std::vector<unsigned int> vlist;
		bool pass;

		if (vertices.size() == 0) return c;

		for (unsigned int f = 0; f < faces.size(); f++)
		{
			for (unsigned int p = 0; p < 3; p++)
			{
				pass = true;
				for (unsigned int v = 0; v < vlist.size(); v++)
				{
					if (vlist[v] == faces[f].point_id[p]) pass = false;
				}
				if (pass) vlist.push_back(faces[f].point_id[p]);
			}
		}

		for (unsigned int i = 0; i < vlist.size(); i++)
		{
			c += vertices[vlist[i]];
		}
		
		c /= (float)vlist.size();

		return c;
	}
	
	void Model::clear()
	{
		meanVertex.set(0, 0, 0);

		normals.clear();
		vertices.clear();
		faces.clear();
		materials.clear();
		texcoords.clear();

	}

	void Model::convex_hull(unsigned int max_faces)
	{
		basecode::Face face;
		bool pass;

		unsigned int pnts[4];
		basecode::vec3f fcentroid;

		do
		{
			faces.clear();
			for (unsigned int i = 0; i < 4; i++)
			{
				pnts[i] = int(basecode::rnd() * vertices.size());
				pass = true;
				for (unsigned int j = 0; j < i; j++)
				{
					if (pnts[j] == pnts[i]) pass = false;
				}
				if (!pass) i -= 1;
			}

			for (unsigned int i = 0; i < 4; i++)
			{
				face.point_id[0] = pnts[i];
				face.point_id[1] = pnts[(i + 1) % 4];
				face.point_id[2] = pnts[(i + 2) % 4];
				newFace(face);
			}

			orient_normals(1);

			pass = true;
			basecode::vec3f n0, n1;
			for (unsigned int a = 0; a < 4; a++)
			{
				n0 = faces[a].normal;
				for (unsigned int b = 1; b < 4; b++)
				{
					n1 = faces[(a + b) % 4].normal;
					if (acos(n0.dot(n1)) > 3.14159f * .7f) pass = false;
					if (!pass) break;
				}
				if (!pass) break;
			}

		} while (!pass);
		
		float maxdist = 0, dist;
		unsigned int maxv;
		unsigned int f;
		std::vector<unsigned int> flist;
		std::vector<basecode::Math::vec2i> edgelist, edge_temp;
		basecode::Math::vec2i edge;

		unsigned int fsize;
		unsigned int fstart;

		int counter = 0;

		do
		{
			counter++;
			if (faces.size() > max_faces) break;
			
			fsize = faces.size();
			fstart = (int)basecode::rnd() * fsize;

			do
			{
				// loop through each face starting from a random place in the stack
				f = (fstart + fsize) % fsize;
				fsize--;

				maxdist = 0;
			
				// find point furthest distance in front of plane
				for (unsigned int v = 0; v < vertices.size(); v++)
				{
					dist = faces[f].normal.dot(vertices[v] - vertices[faces[f].point_id[0]]);
					if (dist <= maxdist) continue;
					maxdist = dist;
					maxv = v;
				}

			} while (fsize > 0 && maxdist == 0);

			
			flist.clear();
			
			// far point found
			if (maxdist > 0)
			{
				// find visible polygons
				edgelist.clear();

				for (unsigned int ff = 0; ff < faces.size(); ff++)
				{
					if (faces[ff].normal.dot(vertices[maxv] - vertices[faces[ff].point_id[0]]) > 0)
					{
						flist.push_back(ff);
						for (unsigned int p = 0; p < 3; p++)
						{
							edge.set(faces[ff].point_id[p], faces[ff].point_id[(p + 1) % 3]);
							pass = true;
							for (unsigned int e = 0; e < edgelist.size(); e++)
							{
								if ((edgelist[e].x == edge.x && edgelist[e].y == edge.y) || (edgelist[e].y == edge.x && edgelist[e].x == edge.y))
								{
									edgelist.erase(edgelist.begin() + e);
									pass = false;
									e--;
								}
							}
							if (pass) edgelist.push_back(edge);
						}
						
					}
				}

				// create boundary loop
				if (flist.size() > 0)
				{
					for (unsigned int e = 0; e < edgelist.size(); e++)
					{
						newFace(edgelist[e].x, edgelist[e].y, maxv);
					}
				}
			}

			if (flist.size() > 0)
			{
				std::vector<basecode::Face> ftemp;

				for (unsigned int f = 0; f < faces.size(); f++)
				{
					pass = true;
					for (unsigned int ff = 0; ff < flist.size(); ff++)
					{
						if (flist[ff] == f) pass = false;
					}

					if ((vertices[faces[f].point_id[0]] - vertices[faces[f].point_id[1]]).magnitude2() < .001f) pass = false;
					if ((vertices[faces[f].point_id[2]] - vertices[faces[f].point_id[1]]).magnitude2() < .001f) pass = false;
					if ((vertices[faces[f].point_id[0]] - vertices[faces[f].point_id[2]]).magnitude2() < .001f) pass = false;
					if (pass) ftemp.push_back(faces[f]);
				}

				faces.clear();
				faces = ftemp;		
			}
			orient_normals(1);
		
		} while (counter < max_faces * 2);
	}


	unsigned int Model::containment_count()
	{
		unsigned int c = vertices.size();
		bool pass;

		for (unsigned int v = 0; v < vertices.size(); v++)
		{
			pass = true;
			for (unsigned int ff = 0; ff < faces.size(); ff++)
			{
				if (faces[ff].normal.dot(vertices[v] - vertices[faces[ff].point_id[0]]) > 0)
				{
					pass = false;
					break;
				}
			}
			if (!pass) c--;
		}

		return c;
	}

	float Model::surface_area()
	{
		float s = 0;
		vec3f n;

		for (unsigned int ff = 0; ff < faces.size(); ff++)
		{
			n = (vertices[faces[ff].point_id[1]] - vertices[faces[ff].point_id[0]]).cross(vertices[faces[ff].point_id[2]] - vertices[faces[ff].point_id[0]]);
			s += n.magnitude() * .5f;
		}

		return s;
	}

	float Model::surface_area2()
	{
		float s = 0;
		vec3f n;

		for (unsigned int ff = 0; ff < faces.size(); ff++)
		{
			n = (vertices[faces[ff].point_id[1]] - vertices[faces[ff].point_id[0]]).cross(vertices[faces[ff].point_id[2]] - vertices[faces[ff].point_id[0]]);
			s += n.dot(n);
		}

		return s;
	}

	void Model::prune_vertices()
	{
		bool pass;
		for (unsigned int v = 0; v < vertices.size(); v++)
		{
			pass = true;
			for (unsigned int f = 0; f < faces.size(); f++)
			{
				if (!pass) break;
				for (unsigned int p = 0; p < 3; p++)
					if (v == faces[f].point_id[p]) pass = false;
			}

			if (pass)
			{
				vertices.erase(vertices.begin() + v);

				for (unsigned int f = 0; f < faces.size(); f++)
				{
					for (unsigned int p = 0; p < 3; p++)
						if (faces[f].point_id[p] > v) faces[f].point_id[p]--;
				}

				v--;
			}
		}
	}

	std::vector<vec2i> Model::edge_list()
	{
		std::vector<vec2i> list;
		vec2i e;
		bool pass;

		for (unsigned int f = 0; f < faces.size(); f++)
		{
			for (unsigned int p = 0; p < 3; p++)
			{
				e.set(faces[f].point_id[p], faces[f].point_id[(p + 1) % 3]);
				pass = true;
				for (unsigned int i = 0; i < list.size(); i++)
				{
					if (list[i].y == faces[f].point_id[p] && list[i].x == faces[f].point_id[(p + 1) % 3]) pass = false;
					if (list[i] == e) pass = false;
				}
				if (pass) list.push_back(e);
			}
		}

		return list;
	}

	void Model::str_explode(std::string haystack, std::string needle, std::vector<std::string> *result)
	{
		std::string::size_type start, end;

		result->clear();

		start = 0;
		end = haystack.find_first_of(needle, start);
		while (end != std::string::npos)
		{
			if (end - start > 0)
			{
				result->push_back(haystack.substr(start, end - start));
			} else {
				result->push_back("");
			}
			start = end + 1;
			end = haystack.find_first_of(needle, start);
		}
		if (end == std::string::npos && haystack.length() - start > 0) result->push_back(haystack.substr(start, haystack.length() - start));
		

	}


	bool Model::load (std::string file)
	{
		clear();

		std::string a;
		if (file.substr(file.length() - 4, 4) == ".obj") return load_obj(file);

		return false;
	}
	

	bool Model::load_obj(std::string file)
	{
		printf("Loading OBJ File '%s'\n", file.c_str());

		std::string directory = file.substr(0, file.find_last_of("/") + 1);
		file = file.substr(file.find_last_of("/") + 1);
		filename = file;
		
		std::fstream fs;

		unsigned int ret;

		char str[1024];
		std::vector<std::string> filelines;

		fs.open((directory + filename).c_str(), std::fstream::in);
		if (!fs.is_open()) return false;

		fs.getline(str, 1024, '\n');
		while (fs.gcount() > 0)
		{
			filelines.push_back(std::string(str));
			fs.getline(str, 1024, '\n');
		}

		fs.close();

		unsigned int totallines = filelines.size();

		std::vector<std::string> expArray;
		std::vector<std::string> expArray2;
		
		std::string curMaterial;
		
		unsigned int vert, norm, uv, tri;

		for (unsigned int i = 0; i < totallines; i++)
		{
			expArray.clear();
			str_explode(filelines[i], " ", &expArray);

			if (expArray.size() > 0)
			{
				if (expArray[0] == "v" && expArray.size() >= 4)
				{
					ret = newVertex_explicit ((float)atof(expArray[1].c_str()), (float)atof(expArray[2].c_str()), (float)atof(expArray[3].c_str()));
				}
				else if (expArray[0] == "vn" && expArray.size() >= 4)
				{
					ret = newNormal ((float)atof(expArray[1].c_str()), (float)atof(expArray[2].c_str()), (float)atof(expArray[3].c_str()));
				}
				else if (expArray[0] == "vt" && expArray.size() == 3)
				{
					ret = newUV ((float)atof(expArray[1].c_str()), (float)atof(expArray[2].c_str()));
				}
				else if (expArray[0] == "mtllib") load_mtl(directory + expArray[1]);
				else if (expArray[0] == "usemtl") curMaterial = expArray[1];
				else if (expArray[0] == "f" && expArray.size() >= 3)
				{
					int npoints = expArray.size() - 1;
					
					tri = newFace();

					faces[tri].materialName = curMaterial;
					faces[tri].npoints = npoints;

					for (int j = 0; j < npoints; j++)
					{
						vert = 0;
						norm = 0;
						uv = 0;

						vert = atoi(expArray[j + 1].c_str());
						if (vert > 0)
						{
							faces[tri].point_id[j] = vert - 1;
						}
					}


					/*int passes = 2;
					int npoints = 4;
					if (expArray.size() == 4) { passes = 1; npoints = 3; }
					if (expArray.size() == 3) { passes = 1; npoints = 2; }

					for (int p = 0; p < passes; p++)
					{
						tri = newFace();

						faces[tri].materialName = curMaterial;
						faces[tri].npoints = npoints;

						for (int j = 0; j < npoints; j++)
						{
							expArray2.clear();
							if (p == 0) str_explode(expArray[j + 1], "/", &expArray2);
							if (p == 1)
							{
								if (j == 0) str_explode(expArray[4], "/", &expArray2);
								if (j == 1) str_explode(expArray[1], "/", &expArray2);
								if (j == 2) str_explode(expArray[3], "/", &expArray2);
							}

							vert = 0;
							norm = 0;
							uv = 0;

							vert = atoi(expArray2[0].c_str());
							if (expArray2.size() > 0) uv = atoi(expArray2[1].c_str());
							if (expArray2.size() > 1) norm = atoi(expArray2[2].c_str());

							if (vert > 0)
							{
								faces[tri].point_id[j] = vert - 1;
								if (norm > 0)
								{
									faces[tri].vnormals[j] = normals[norm - 1];
									faces[tri].normal = normals[norm - 1];
								}
								if (uv <= texcoords.size() && texcoords.size() > 0) faces[tri].uvs[j] = texcoords[uv - 1];
							}
						}
					}*/

				}

			}
			
		}

		meanVertex.set(0, 0, 0);
		
		for (unsigned int v = 0; v < vertices.size(); v++)
			meanVertex += vertices[v];

		if (vertices.size() > 0) meanVertex /= (float)vertices.size();	

		return true;
	}


	void Model::load_mtl(std::string file)
	{
		printf("Loading MTL File '%s'\n", file.c_str());

		std::fstream fs;

		char str[1024];
		std::vector<std::string> filelines;

		fs.open(file.c_str(), std::fstream::in);
		if (!fs.is_open()) return;

		std::string fline;

		fs.getline(str, 1024, '\n');
		while (fs.gcount() > 0)
		{
			fline = std::string(str);
			std::remove(fline.begin(), fline.end(), 9);		// strip tabs
			filelines.push_back(fline);
			fs.getline(str, 1024, '\n');
		}

		fs.close();

		unsigned int totallines = filelines.size();

		unsigned int curMaterial = -1;
		Material material;

		std::vector<std::string> expArray;
		std::vector<std::string> expArray2;
		
		for (unsigned int i = 0; i < totallines; i++)
		{
			expArray.clear();
			str_explode(filelines[i], " ", &expArray);

			if (expArray.size() > 0)
			{
				if (expArray[0] == "newmtl")
				{
					material.name = expArray[1];
					materials.push_back(material);
					curMaterial++;
				}
				else if (expArray[0] == "map_Kd") materials[curMaterial].textureStr = expArray[1];
				else if (expArray[0] == "Kd") materials[curMaterial].color.set((float)atof(expArray[1].c_str()), (float)atof(expArray[2].c_str()), (float)atof(expArray[3].c_str()));
				else if (expArray[0] == "Ke") materials[curMaterial].emitting.set((float)atof(expArray[1].c_str()), (float)atof(expArray[2].c_str()), (float)atof(expArray[3].c_str()));
			}
		}
	}


	unsigned int Model::newNormal (float x, float y, float z)
	{
		normals.push_back(vec3f(x, y, z));

		return normals.size() - 1;
	}
	

	unsigned int Model::newNormal (vec3f n)
	{
		return newNormal(n.x, n.y, n.z);
	}
	

	unsigned int Model::newFace ()
	{
		faces.push_back(Face());
		
		return faces.size() - 1;
	}
	

	unsigned int Model::newFace (vec3f p1, vec3f p2, vec3f p3)
	{
		unsigned int tri = newFace();

		faces[tri].point_id[0] = newVertex(p1);
		faces[tri].point_id[1] = newVertex(p2);
		faces[tri].point_id[2] = newVertex(p3);

		faces[tri].normal = (p2 - p1).cross(p3 - p1);
		faces[tri].normal.normalize();
		faces[tri].d = faces[tri].normal.dot(p1);

		return tri;
	}

	unsigned int Model::newFace (vec3f p1, vec3f p2, vec3f p3, Color c)
	{
		unsigned int tri = newFace();

		faces[tri].point_id[0] = newVertex(p1);
		faces[tri].point_id[1] = newVertex(p2);
		faces[tri].point_id[2] = newVertex(p3);

		faces[tri].normal = (p2 - p1).cross(p3 - p1);
		faces[tri].normal.normalize();
		faces[tri].d = faces[tri].normal.dot(p1);

		faces[tri].vcolors[0] = c;
		faces[tri].vcolors[1] = c;
		faces[tri].vcolors[2] = c;
		
		return tri;
	}

	unsigned int Model::newFace (unsigned int p1, unsigned int p2, unsigned int p3)
	{
		unsigned int tri = newFace();

		faces[tri].point_id[0] = p1;
		faces[tri].point_id[1] = p2;
		faces[tri].point_id[2] = p3;

		basecode::vec3f v1 = vertices[p1];
		basecode::vec3f v2 = vertices[p2];
		basecode::vec3f v3 = vertices[p3];

		faces[tri].normal = (v2 - v1).cross(v3 - v1);
		faces[tri].normal.normalize();
		faces[tri].d = faces[tri].normal.dot(v1);


		return tri;
	}
	

	unsigned int Model::newFace (Face t)
	{
		unsigned int tri = newFace();

		faces[tri] = t;

		basecode::vec3f v1 = vertices[t.point_id[0]];
		basecode::vec3f v2 = vertices[t.point_id[1]];
		basecode::vec3f v3 = vertices[t.point_id[2]];

		faces[tri].normal = (v2 - v1).cross(v3 - v1);
		faces[tri].normal.normalize();
		faces[tri].d = faces[tri].normal.dot(v1);

		return tri;
	}
	

	unsigned int Model::newFace_explicit (vec3f p1, vec3f p2, vec3f p3)
	{
		unsigned int tri = newFace();

		faces[tri].point_id[0] = newVertex_explicit(p1);
		faces[tri].point_id[1] = newVertex_explicit(p2);
		faces[tri].point_id[2] = newVertex_explicit(p3);

		faces[tri].normal = (p2 - p1).cross(p3 - p1);
		faces[tri].normal.normalize();
		faces[tri].d = faces[tri].normal.dot(p1);


		return tri;
	}


	unsigned int Model::newFace_explicit (vec3f p1, vec3f p2, vec3f p3, Color c)
	{
		unsigned int tri = newFace();

		faces[tri].point_id[0] = newVertex_explicit(p1);
		faces[tri].point_id[1] = newVertex_explicit(p2);
		faces[tri].point_id[2] = newVertex_explicit(p3);

		faces[tri].normal = (p2 - p1).cross(p3 - p1);
		faces[tri].normal.normalize();
		faces[tri].d = faces[tri].normal.dot(p1);


		faces[tri].vcolors[0] = c;
		faces[tri].vcolors[1] = c;
		faces[tri].vcolors[2] = c;
		
		return tri;
	}


	unsigned int Model::newUV (float u, float v)
	{
		texcoords.push_back(vec2f(u, v));

		return texcoords.size() - 1;
	}
	

	unsigned int Model::newUV (vec2f uv)
	{
		return newUV(uv.x, uv.y);
	}

	unsigned int Model::newVertex (float x, float y, float z)
	{
		for (unsigned int i = 0; i < vertices.size(); i++)
			if (vertices[i].x - vertexSnapDistance <= x && vertices[i].x + vertexSnapDistance >= x)
				if (vertices[i].y - vertexSnapDistance <= y && vertices[i].y + vertexSnapDistance >= y)
					if (vertices[i].z - vertexSnapDistance <= z && vertices[i].z + vertexSnapDistance >= z)
						return i;

		return newVertex_explicit(x, y, z);
	}


	unsigned int Model::newVertex (vec3f v)
	{
		return newVertex(v.x, v.y, v.z);
	}


	unsigned int Model::newVertex_explicit (float x, float y, float z)
	{
		vertices.push_back(vec3f(x, y, z));
		
		return vertices.size() - 1;
	}


	unsigned int Model::newVertex_explicit (vec3f v)
	{
		return newVertex_explicit(v.x, v.y, v.z);
	}


	void Model::normalize()
	{
		float max_dist = 0;

		for (unsigned int i = 0; i < vertices.size(); i++)
			if (vertices[i].magnitude() > max_dist) max_dist = vertices[i].magnitude();
		
		for (unsigned int i = 0; i < vertices.size(); i++)
			vertices[i] /= max_dist;

		meanVertex /= max_dist;
	}


	void Model::orient_normals(float direction)
	{
		vec3f c = fcentroid();

		for (unsigned int f = 0; f < faces.size(); f++)
		{
			if (faces[f].normal.dot(vertices[faces[f].point_id[0]] - c) * sgn(direction) < 0)
			{
				unsigned int a = faces[f].point_id[2];
				faces[f].point_id[2] = faces[f].point_id[1];
				faces[f].point_id[1] = a;

				calc_normal(f);
			}
		}
	}

	float Model::radius()
	{
		float maxdist2 = 0;
		float dist2;

		for (unsigned int v = 0; v < vertices.size(); v++)
		{
			dist2 = vertices[v].magnitude2();
			if (dist2 > maxdist2) maxdist2 = dist2;
		}

		return sqrt(maxdist2);
	}

	float Model::minRadius()
	{
		float mx = 0, my = 0, mz = 0;
		float nx, ny, nz;

		if (vertices.size() == 0) return 0;

		for (unsigned int v = 0; v < vertices.size(); v++)
		{
			nx = fabs(vertices[v].x);
			ny = fabs(vertices[v].y);
			nz = fabs(vertices[v].z);

			if (v == 0) { mx = nx; my = ny; mz = nz; }

			if (nx > mx) mx = nx;
			if (ny > my) my = ny;
			if (nz > mz) mz = nz;
		}

		if (my < mx && my < mz) return my;
		if (mz < mx && mz < my) return mz;
		return nx;
	}


	void Model::removeCreases()
	{
		bool pass = false;
		bool clip = false;
		for (unsigned int f = 0; f < faces.size(); f++)
		{
			clip = false;
			if (faces[f].point_id[0] == faces[f].point_id[1]) clip = true;
			if (faces[f].point_id[0] == faces[f].point_id[2]) clip = true;
			if (faces[f].point_id[2] == faces[f].point_id[1]) clip = true;

			if (clip)
			{
				faces.erase(faces.begin() + f);
				pass = true;
			}
		}
		prune_vertices();
	}

	float Model::getVertexSnapDistance()
	{
		return vertexSnapDistance;
	}

	
	void Model::setVertexSnapDistance(float d)
	{
		vertexSnapDistance = d;
	}

	
	void Model::scale(float x0, float y0, float z0)
	{
		for (unsigned int i = 0; i < vertices.size(); i++)
		{
			vertices[i].x *= x0;
			vertices[i].y *= y0;
			vertices[i].z *= z0;
		}

		meanVertex.x *= x0;
		meanVertex.y *= y0;
		meanVertex.z *= z0;
	}

	void Model::scale(float s0)
	{
		scale(s0, s0, s0);
	}


	Model* ModelHandler::newModel()
	{
		printf("New Model\n");
		modelList.push_back(Model());
		return &*(modelList.rbegin());
	}

	void ModelHandler::deleteModel(Model* model)
	{
		printf("Delete Model\n");
		std::list<Model>::iterator it, found;
		bool success = false;

		for(it = modelList.begin(); it != modelList.end(); it++)
		{
			if (model == &*(it))
			{
				found = it;
				success = true;
			}
		}
		if (success) modelList.erase(found);

	}

	void Model::genBuffers()
	{
		glGenBuffers(1, &vertexBuffer);
		glGenBuffers(1, &colorBuffer);
	}

	void Model::buildBuffers()
	{
		std::vector<float> vtx;
		
		for (unsigned int f = 0; f < faces.size(); f++)
			for (unsigned int i = 0; i < 3; i++)
			{
				vtx.push_back(vertices[faces[f].point_id[i]].x);
				vtx.push_back(vertices[faces[f].point_id[i]].y);
				vtx.push_back(vertices[faces[f].point_id[i]].z);
			}

		glBindBuffer(GL_ARRAY_BUFFER, vertexBuffer);
		glBufferData(GL_ARRAY_BUFFER, vtx.size() * sizeof(float), &vtx[0], GL_STATIC_DRAW);	
	}

	void Model::color(float r, float g, float b)
	{
		std::vector<float> clr;
		float ratio;
		
		for (unsigned int f = 0; f < faces.size(); f++)
		{
			ratio = .5f + .5f * faces[f].normal.dot(0, 1, 0);

			for (unsigned int i = 0; i < 3; i++)
			{
				clr.push_back(r * ratio);
				clr.push_back(g * ratio);
				clr.push_back(b * ratio);
			}
		}

		glBindBuffer(GL_ARRAY_BUFFER, colorBuffer);
		glBufferData(GL_ARRAY_BUFFER, clr.size() * sizeof(float), &clr[0], GL_STATIC_DRAW);	
	}
}

